[**DeadZone Community Packages**](README.md)

***

[DeadZone Community Packages](README.md) / Banking Example

# Basic Banking Example
## About
As a small example, let's write a package which withdraws some items from the bank.

Our package should do the following
- Open the bank
- Check if they have the required item in the bank
- Withdraw the required quantity
- Close the bank

## Open the bank
Let's start with **OnGameTick** with the following changes

Looking at the API we can use [Game.info.bank.isOpen](DeadZoneAPI.Game.info.bank.Function.isOpen.md)

```
function OnGameTick() {
    // Increment our total ticks running by one
    totalTicksRunning++;

    if(!Game.info.bank.isOpen())
    {
        Game.interact.bank.openNearest(0, true);
    }
}
```

If you give this code a run, you'll see your character run over to the bank and open the interface.

## Withdraw
### Adding a config item
Now let's withdraw an item! We should make this configurable so someone can choose which item to withdraw. 

Navigate to the 'config' tab.

From here we can add in a new configuration item. Since items ingame all work as 'Item IDs' we want to create a new Integer configuration.

```
const config = {
    itemToWithdraw: ConfigItem.createInteger("itemToWithdraw", "Banking", "Item to withdraw", "This is what we will take out of the bank", 1511);
};
```

It would be good to type out the above, rather than copy as you will see the autocomplete which describes the purpose of each value.

![Configuration Editing](_media/configItem.png "Configuration Editing")

Now we have created our configuration value, we can use it in our main code.

### Checking the bank
Update the code to use [Game.info.bank.hasItem](DeadZoneAPI.Game.info.bank.Function.hasItem.md) to see if we have the item in our bank. If not, we should stop the script there.

```
function OnGameTick() {
    // Increment our total ticks running by one
    totalTicksRunning++;

    if(!Game.info.bank.isOpen())
    {
        Game.interact.bank.openNearest(0, true);
    }
    else
    {
        // The bank is open. But do we have the items?
        if( Game.info.bank.hasItem( config.itemToWithdraw.getValue(), 5) )
        {
            // We want to have a minimum of 5 available in the bank to continue
        }
        else
        {
            Game.sendGameMessage("Error! Ran out of items!", "Bank Withdraw Package");
            Utility.packages.shutdown();
        }
    }
}
```

You can see that if we do not have the item we will print an error message in chat, then shutdown.

### Withdraw action
We can now try and withdraw the item with [Game.interact.bank.withdrawItem](DeadZoneAPI.Game.interact.bank.Function.withdrawItem.md). Notice in the documentation that this takes a parameter for [BankWithdrawType](DeadZoneAPI.Enumeration.BankWithdrawType.md). We've made it easier by having some function calls pass in pre-made ways to carry out actions.

```
// The bank is open. But do we have the items?
if( Game.info.bank.hasItem( config.itemToWithdraw.getValue(), 5) )
{
    // We want to have a minimum of 5 available in the bank to continue
    Game.interact.bank.withdrawItem( config.itemToWithdraw.getValue(), BankWithdrawType.WITHDRAW_FIVE, false );
}
```

### Inventory checks
Running the script you'll notice it will continue to withdraw, and won't stop!

We need to put some checks in to stop when we have the required items. We should check this each tick, before doing any actions. This can be done with [Game.info.inventory.hasItem](DeadZoneAPI.Game.info.inventory.Function.hasItem.md)

```
function OnGameTick() {
    // Increment our total ticks running by one
    totalTicksRunning++;

    if( Game.info.inventory.hasItem( config.itemToWithdraw.getValue(), 5 ) )
    {
        if( Game.info.bank.isOpen() )
        {
            Game.interact.bank.close();
        }

        Game.sendGameMessage("You now have the items you require!", "Bank Withdraw Package");
        Utility.packages.shutdown();
    }

    if(!Game.info.bank.isOpen())
    {
        Game.interact.bank.openNearest(0, true);
    }

    ...
}
```

## Delays
Note that this is a very basic example which shows you how the OnGameTick operates and how you can structure and piece together your package. What is has not covered is applying delays to each of the actions.

Each tick is 0.6 seconds, so our actions will be executing **exactly** on the start of a tick, it's a bit suspicious!

What we need to do instead if apply a random delay. You can use the API calls to handle this for you.

```
Utility.invokeLater(function () {
    Game.interact.bank.openNearest(0, true);
}, Utility.getDelay());
```

Each call to an action should be wrapped in an invokeLater, passing in a delay. The delay is automatically generated using the user's profile in DZ API.
